$(document).ready(function() {

    $('.smart-editor').append(`<div class="toolbar">
        <div class="toolbar-group">
            <button id="boldBtn" title="Bold"><i class="fas fa-bold"></i></button>
            <button id="italicBtn" title="Italic"><i class="fas fa-italic"></i></button>
            <button id="underlineBtn" title="Underline"><i class="fas fa-underline"></i></button>
            <button id="strikethroughBtn" title="Strikethrough"><i class="fas fa-strikethrough"></i></button>
        </div>
        <div class="toolbar-group">
            <select id="headingSelect">
                <option value="">Paragraph</option>
                <option value="h1">Heading 1</option>
                <option value="h2">Heading 2</option>
                <option value="h3">Heading 3</option>
                <option value="h4">Heading 4</option>
            </select>
            <select id="fontSelect">
                <option value="Arial, sans-serif">Arial</option>
                <option value="'Times New Roman', serif">Times New Roman</option>
                <option value="'Courier New', monospace">Courier New</option>
                <option value="Georgia, serif">Georgia</option>
                <option value="Verdana, sans-serif">Verdana</option>
            </select>
            <select id="fontSizeSelect">
                <option value="1">8pt</option>
                <option value="2">10pt</option>
                <option value="3" selected>12pt</option>
                <option value="4">14pt</option>
                <option value="5">18pt</option>
                <option value="6">24pt</option>
                <option value="7">36pt</option>
            </select>
        </div>
        <div class="toolbar-group">
            <button id="alignLeftBtn" title="Align Left"><i class="fas fa-align-left"></i></button>
            <button id="alignCenterBtn" title="Align Center"><i class="fas fa-align-center"></i></button>
            <button id="alignRightBtn" title="Align Right"><i class="fas fa-align-right"></i></button>
            <button id="alignJustifyBtn" title="Justify"><i class="fas fa-align-justify"></i></button>
        </div>
        <div class="toolbar-group">
            <button id="orderedListBtn" title="Ordered List"><i class="fas fa-list-ol"></i></button>
            <button id="unorderedListBtn" title="Unordered List"><i class="fas fa-list-ul"></i></button>
            <button id="indentBtn" title="Indent"><i class="fas fa-indent"></i></button>
            <button id="outdentBtn" title="Outdent"><i class="fas fa-outdent"></i></button>
        </div>
        <div class="toolbar-group">
            <button id="linkBtn" title="Insert Link"><i class="fas fa-link"></i></button>
            <button id="unlinkBtn" title="Remove Link"><i class="fas fa-unlink"></i></button>
            <button id="imageBtn" title="Insert Image"><i class="fas fa-image"></i></button>
            <button id="tableBtn" title="Insert Table"><i class="fas fa-table"></i></button>
            <button id="mathBtn" title="Insert Math Equation"><i class="fas fa-square-root-alt"></i></button>
        </div>
        <div class="toolbar-group">
            <input type="color" id="foreColorInput" title="Text Color">
            <input type="color" id="backColorInput" title="Background Color" value="#ffffff">
        </div>
        <div class="toolbar-group">
            <button id="undoBtn" title="Undo"><i class="fas fa-undo"></i></button>
            <button id="redoBtn" title="Redo"><i class="fas fa-redo"></i></button>
            <button id="clearFormatBtn" title="Clear Formatting"><i class="fas fa-eraser"></i></button>
        </div>
    </div>
    <div class="editor-content">
        <div id="editor" class="content-area" contenteditable="true"></div>
    </div>
    <div class="status-bar">
        <div id="wordCount">Words: 0 | Characters: 0</div>
        <div id="editorMode">Design Mode</div>
    </div>
    <!-- Link Dialog -->
    <div id="linkDialog" class="dialog-overlay">
        <div class="dialog">
            <div class="dialog-header">
                <div class="dialog-title">Insert Link</div>
                <button class="dialog-close" id="closeLinkDialog">&times;</button>
            </div>
            <div class="dialog-content">
                <div class="form-group">
                    <label for="linkText">Text to display:</label>
                    <input type="text" id="linkText" class="form-control">
                </div>
                <div class="form-group">
                    <label for="linkUrl">URL:</label>
                    <input type="text" id="linkUrl" class="form-control" placeholder="https://">
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" id="linkNewTab"> Open in new tab
                    </label>
                </div>
            </div>
            <div class="dialog-footer">
                <button class="btn btn-secondary" id="cancelLinkBtn">Cancel</button>
                <button class="btn btn-primary" id="insertLinkBtn">Insert</button>
            </div>
        </div>
    </div>

    <!-- Image Dialog -->
    <div id="imageDialog" class="dialog-overlay">
        <div class="dialog">
            <div class="dialog-header">
                <div class="dialog-title">Insert Image</div>
                <button class="dialog-close" id="closeImageDialog">&times;</button>
            </div>
            <div class="dialog-content">
                <div class="form-group">
                    <label for="imageUrl">Image URL:</label>
                    <input type="text" id="imageUrl" class="form-control" placeholder="https://">
                </div>
                <div class="form-group">
                    <label for="imageAlt">Alternative text:</label>
                    <input type="text" id="imageAlt" class="form-control" placeholder="Image description">
                </div>
                <div class="form-group">
                    <label for="imageWidth">Width (px):</label>
                    <input type="number" id="imageWidth" class="form-control">
                </div>
                <div class="form-group">
                    <label for="imageHeight">Height (px):</label>
                    <input type="number" id="imageHeight" class="form-control">
                </div>
            </div>
            <div class="dialog-footer">
                <button class="btn btn-secondary" id="cancelImageBtn">Cancel</button>
                <button class="btn btn-primary" id="insertImageBtn">Insert</button>
            </div>
        </div>
    </div>

    <!-- Table Dialog -->
    <div id="tableDialog" class="dialog-overlay">
        <div class="dialog">
            <div class="dialog-header">
                <div class="dialog-title">Insert Table</div>
                <button class="dialog-close" id="closeTableDialog">&times;</button>
            </div>
            <div class="dialog-content">
                <div class="form-group">
                    <label for="tableRows">Rows:</label>
                    <input type="number" id="tableRows" class="form-control" value="2" min="1" max="20">
                </div>
                <div class="form-group">
                    <label for="tableColumns">Columns:</label>
                    <input type="number" id="tableColumns" class="form-control" value="2" min="1" max="10">
                </div>
                <div class="form-group">
                    <label for="tableWidth">Width:</label>
                    <select id="tableWidth" class="form-control">
                        <option value="100%">100%</option>
                        <option value="75%">75%</option>
                        <option value="50%">50%</option>
                        <option value="custom">Custom</option>
                    </select>
                </div>
                <div class="form-group" id="customWidthGroup" style="display: none;">
                    <label for="customTableWidth">Custom width:</label>
                    <input type="text" id="customTableWidth" class="form-control" placeholder="e.g., 400px">
                </div>
            </div>
            <div class="dialog-footer">
                <button class="btn btn-secondary" id="cancelTableBtn">Cancel</button>
                <button class="btn btn-primary" id="insertTableBtn">Insert</button>
            </div>
        </div>
    </div>

    <!-- Math Dialog -->
    <div id="mathDialog" class="dialog-overlay">
        <div class="dialog">
            <div class="dialog-header">
                <div class="dialog-title">Insert Math Equation </div>
                <button class="dialog-close" id="closeMathDialog">&times;</button>
            </div>
            <div class="dialog-content">
                <div class="form-group">
                    <label for="mathEquation">LaTeX Equation:</label>
                    <textarea id="mathEquation" class="form-control" rows="4" placeholder="e.g., E = mc^2"></textarea>
                </div>
                <div class="form-group">
                    <label>Display Mode:</label>
                    <div>
                        <label>
                            <input type="radio" name="mathMode" value="inline" checked> Inline (\( ... \))
                        </label>
                    </div>
                    <div>
                        <label>
                            <input type="radio" name="mathMode" value="block"> Block (\[ ... \])
                        </label>
                    </div>
                </div>
                <div class="form-group">
                    <label>Preview:</label>
                    <div id="mathPreview" style="min-height: 50px; padding: 8px; border: 1px solid #ddd; border-radius: 4px;"></div>
                </div>
            </div>
            <div class="dialog-footer">
                <button class="btn btn-secondary" id="cancelMathBtn">Cancel</button>
                <button class="btn btn-primary" id="insertMathBtn">Insert</button>
            </div>
        </div>
    </div>`);

    // Initialize MathJax
    MathJax.typeset();

    // Track editor state
    let editorState = {
        currentSelection: null,
        selectedImage: null,
        selectedTable: null,
        undoStack: [],
        redoStack: [],
        lastHtml: ''
    };

    // Make editor beautiful when class is added
     $(document).ready(function() {
        $('.smart-editor').addClass('beautiful-editor');
    });

    // Initialize editor
    const editor = document.getElementById('editor');
    editor.innerHTML = '<p></p>';
    editor.focus();
    
    // Save initial state for undo
    saveCurrentState();

    // Execute command function
    function execCommand(command, value = null) {
        document.execCommand(command, false, value);
        updateButtonStates();
        saveCurrentState();
    }

    // Update toolbar button states based on current selection
    function updateButtonStates() {
        $('#boldBtn').toggleClass('active', document.queryCommandState('bold'));
        $('#italicBtn').toggleClass('active', document.queryCommandState('italic'));
        $('#underlineBtn').toggleClass('active', document.queryCommandState('underline'));
        $('#strikethroughBtn').toggleClass('active', document.queryCommandState('strikeThrough'));
        $('#alignLeftBtn').toggleClass('active', document.queryCommandState('justifyLeft'));
        $('#alignCenterBtn').toggleClass('active', document.queryCommandState('justifyCenter'));
        $('#alignRightBtn').toggleClass('active', document.queryCommandState('justifyRight'));
        $('#alignJustifyBtn').toggleClass('active', document.queryCommandState('justifyFull'));
        $('#orderedListBtn').toggleClass('active', document.queryCommandState('insertOrderedList'));
        $('#unorderedListBtn').toggleClass('active', document.queryCommandState('insertUnorderedList'));
    }

    // Save editor state for undo/redo
    function saveCurrentState() {
        const currentHtml = editor.innerHTML;
        if (currentHtml !== editorState.lastHtml) {
            editorState.undoStack.push(editorState.lastHtml);
            editorState.redoStack = [];
            editorState.lastHtml = currentHtml;
            
            // Limit undo stack size
            if (editorState.undoStack.length > 50) {
                editorState.undoStack.shift();
            }
            
            updateWordCount();
        }
    }

    // Update word count
    function updateWordCount() {
        const text = editor.innerText || '';
        const words = text.trim() ? text.trim().split(/\s+/).length : 0;
        const chars = text.length;
        $('#wordCount').text(`Words: ${words} | Characters: ${chars}`);
    }

    // Toolbar button click handlers
    $('#boldBtn').on('click', function() { execCommand('bold'); });
    $('#italicBtn').on('click', function() { execCommand('italic'); });
    $('#underlineBtn').on('click', function() { execCommand('underline'); });
    $('#strikethroughBtn').on('click', function() { execCommand('strikeThrough'); });
    $('#alignLeftBtn').on('click', function() { execCommand('justifyLeft'); });
    $('#alignCenterBtn').on('click', function() { execCommand('justifyCenter'); });
    $('#alignRightBtn').on('click', function() { execCommand('justifyRight'); });
    $('#alignJustifyBtn').on('click', function() { execCommand('justifyFull'); });
    $('#orderedListBtn').on('click', function() { execCommand('insertOrderedList'); });
    $('#unorderedListBtn').on('click', function() { execCommand('insertUnorderedList'); });
    $('#indentBtn').on('click', function() { execCommand('indent'); });
    $('#outdentBtn').on('click', function() { execCommand('outdent'); });
    $('#unlinkBtn').on('click', function() { execCommand('unlink'); });
    $('#clearFormatBtn').on('click', function() { execCommand('removeFormat'); });

    // Undo and Redo
    $('#undoBtn').on('click', function() {
        if (editorState.undoStack.length > 0) {
            const prevState = editorState.undoStack.pop();
            editorState.redoStack.push(editorState.lastHtml);
            editor.innerHTML = prevState;
            editorState.lastHtml = prevState;
            updateWordCount();
        }
    });

    $('#redoBtn').on('click', function() {
        if (editorState.redoStack.length > 0) {
            const nextState = editorState.redoStack.pop();
            editorState.undoStack.push(editorState.lastHtml);
            editor.innerHTML = nextState;
            editorState.lastHtml = nextState;
            updateWordCount();
        }
    });

    // Heading change
    $('#headingSelect').on('change', function() {
        const value = $(this).val();
        if (value) {
            execCommand('formatBlock', `<${value}>`);
        } else {
            execCommand('formatBlock', '<p>');
        }
    });

    // Font family change
    $('#fontSelect').on('change', function() {
        execCommand('fontName', $(this).val());
    });

    // Font size change
    $('#fontSizeSelect').on('change', function() {
        execCommand('fontSize', $(this).val());
    });

    // Text color change
    $('#foreColorInput').on('change', function() {
        execCommand('foreColor', $(this).val());
    });

    // Background color change
    $('#backColorInput').on('change', function() {
        execCommand('hiliteColor', $(this).val());
    });

    // Link handling
    $('#linkBtn').on('click', function() {
        saveSelection();
        const selection = document.getSelection();
        $('#linkText').val(selection.toString());
        $('#linkDialog').show();
    });

    $('#closeLinkDialog, #cancelLinkBtn').on('click', function() {
        $('#linkDialog').hide();
    });

    $('#insertLinkBtn').on('click', function() {
        const url = $('#linkUrl').val();
        const text = $('#linkText').val();
        const newTab = $('#linkNewTab').is(':checked') ? ' target="_blank"' : '';
        
        restoreSelection();
        
        if (url) {
            if (text && document.getSelection().toString() !== text) {
                document.execCommand('insertHTML', false, `<a href="${url}"${newTab}>${text}</a>`);
            } else {
                document.execCommand('createLink', false, url);
                if (newTab) {
                    const selection = document.getSelection();
                    if (selection.rangeCount > 0) {
                        const range = selection.getRangeAt(0);
                        const link = range.startContainer.parentNode;
                        if (link.tagName === 'A') {
                            link.setAttribute('target', '_blank');
                        }
                    }
                }
            }
            saveCurrentState();
        }
        
        $('#linkDialog').hide();
    });

    // Image handling
    $('#imageBtn').on('click', function() {
        saveSelection();
        $('#imageDialog').show();
    });

    $('#closeImageDialog, #cancelImageBtn').on('click', function() {
        $('#imageDialog').hide();
    });

    $('#insertImageBtn').on('click', function() {
        const url = $('#imageUrl').val();
        const alt = $('#imageAlt').val();
        const width = $('#imageWidth').val() ? ` width="${$('#imageWidth').val()}"` : '';
        const height = $('#imageHeight').val() ? ` height="${$('#imageHeight').val()}"` : '';
        
        restoreSelection();
        
        if (url) {
            const imgHtml = `<div class="image-wrapper" draggable="true">
                <img src="${url}" alt="${alt}"${width}${height} draggable="false">
                <div class="image-controls">
                    <button class="resize-img-btn" title="Resize"><i class="fas fa-expand"></i></button>
                    <button class="delete-img-btn" title="Delete"><i class="fas fa-trash"></i></button>
                </div>
                <div class="resize-handle"></div>
            </div>`;
            
            document.execCommand('insertHTML', false, imgHtml);
            setupImageHandlers();
            saveCurrentState();
        }
        
        $('#imageDialog').hide();
        $('#imageUrl, #imageAlt, #imageWidth, #imageHeight').val('');
    });

    // Table handling
    $('#tableBtn').on('click', function() {
        saveSelection();
        $('#tableDialog').show();
    });

    $('#closeTableDialog, #cancelTableBtn').on('click', function() {
        $('#tableDialog').hide();
    });

    $('#tableWidth').on('change', function() {
if ($(this).val() === 'custom') {
$('#customWidthGroup').show();
} else {
$('#customWidthGroup').hide();
}
});

$('#insertTableBtn').on('click', function() {
const rows = parseInt($('#tableRows').val()) || 2;
const cols = parseInt($('#tableColumns').val()) || 2;
let width = $('#tableWidth').val();

if (width === 'custom') {
width = $('#customTableWidth').val() || '100%';
}

restoreSelection();

let tableHtml = `<table style="width: ${width};">`;
tableHtml += '<thead><tr>';

for (let i = 0; i < cols; i++) {
tableHtml += '<th>Header ' + (i + 1) + '<div class="resizer"></div></th>';
}

tableHtml += '</tr></thead><tbody>';

for (let i = 0; i < rows - 1; i++) {
tableHtml += '<tr>';
for (let j = 0; j < cols; j++) {
    tableHtml += '<td>Cell ' + (i + 1) + '-' + (j + 1) + '<div class="resizer"></div></td>';
}
tableHtml += '</tr>';
}

tableHtml += '</tbody></table>';

document.execCommand('insertHTML', false, tableHtml);
setupTableHandlers();
saveCurrentState();

$('#tableDialog').hide();
});

// Math equation handling
$('#mathBtn').on('click', function() {
saveSelection();
$('#mathDialog').show();
updateMathPreview();
});

$('#closeMathDialog, #cancelMathBtn').on('click', function() {
$('#mathDialog').hide();
});

$('#mathEquation').on('input', function() {
updateMathPreview();
});

$('input[name="mathMode"]').on('change', function() {
updateMathPreview();
});

function updateMathPreview() {
const equation = $('#mathEquation').val();
const mode = $('input[name="mathMode"]:checked').val();

if (equation) {
const mathMarkup = mode === 'inline' ? 
    `\\(${equation}\\)` : 
    `\\[${equation}\\]`;

$('#mathPreview').html(mathMarkup);
MathJax.typesetPromise([$('#mathPreview')[0]]).catch(function(err) {
    console.log('MathJax error:', err);
    $('#mathPreview').html('<span style="color: red;">Error in equation</span>');
});
} else {
$('#mathPreview').html('');
}
}

$('#insertMathBtn').on('click', function() {
const equation = $('#mathEquation').val();
const mode = $('input[name="mathMode"]:checked').val();

if (equation) {
restoreSelection();

const mathMarkup = mode === 'inline' ? 
    `<span class="math-equation" data-mode="inline">\\(${equation}\\)</span>` : 
    `<div class="math-equation" data-mode="block">\\[${equation}\\]</div>`;

document.execCommand('insertHTML', false, mathMarkup);
MathJax.typesetPromise([editor]).catch(function(err) {
    console.log('MathJax error:', err);
});
saveCurrentState();
}

$('#mathDialog').hide();
$('#mathEquation').val('');
});

// Save and restore selection
function saveSelection() {
editorState.currentSelection = null;
const selection = window.getSelection();
if (selection.rangeCount > 0) {
editorState.currentSelection = selection.getRangeAt(0).cloneRange();
}
}

function restoreSelection() {
if (editorState.currentSelection) {
const selection = window.getSelection();
selection.removeAllRanges();
selection.addRange(editorState.currentSelection);
editor.focus();
}
}

// Setup image handling
function setupImageHandlers() {
$('.image-wrapper').off().on('click', function(e) {
e.stopPropagation();
$('.image-wrapper').removeClass('active');
$(this).addClass('active');
editorState.selectedImage = this;
});

$('.resize-img-btn').off().on('click', function(e) {
e.stopPropagation();
const wrapper = $(this).closest('.image-wrapper');
const img = wrapper.find('img');

$('#imageUrl').val(img.attr('src'));
$('#imageAlt').val(img.attr('alt'));
$('#imageWidth').val(img.attr('width'));
$('#imageHeight').val(img.attr('height'));

editorState.selectedImage = wrapper[0];
$('#imageDialog').show();
});

$('.delete-img-btn').off().on('click', function(e) {
e.stopPropagation();
$(this).closest('.image-wrapper').remove();
saveCurrentState();
});

// Make images resizable and draggable
$('.image-wrapper').draggable({
containment: editor,
handle: '.move-handle',
start: function() {
    $(this).addClass('active');
},
stop: function() {
    saveCurrentState();
}
});

$('.resize-handle').draggable({
containment: editor,
start: function(e, ui) {
    const wrapper = $(this).parent();
    const img = wrapper.find('img');
    this._startWidth = img.width();
    this._startHeight = img.height();
    this._startPosition = { left: ui.position.left, top: ui.position.top };
},
drag: function(e, ui) {
    const wrapper = $(this).parent();
    const img = wrapper.find('img');
    const deltaX = ui.position.left - this._startPosition.left;
    const deltaY = ui.position.top - this._startPosition.top;
    
    img.width(this._startWidth + deltaX);
    img.height(this._startHeight + deltaY);
},
stop: function() {
    saveCurrentState();
}
});
}

// Setup table handlers
function setupTableHandlers() {
$('.resizer').off().on('mousedown', function(e) {
e.preventDefault();
const th = $(this).parent();
const table = th.closest('table');
const initialX = e.pageX;
const initialWidth = th.width();

$(document).on('mousemove', function(e) {
    const width = initialWidth + (e.pageX - initialX);
    th.width(width);
});

$(document).on('mouseup', function() {
    $(document).off('mousemove mouseup');
    saveCurrentState();
});
});
}

// Editor event listeners
$(editor).on('mouseup keyup', function() {
updateButtonStates();
saveCurrentState();
});

$(editor).on('click', function(e) {
if ($(e.target).closest('.image-wrapper').length === 0) {
$('.image-wrapper').removeClass('active');
}
});

// Initialize the editor
setupImageHandlers();
setupTableHandlers();

// Listen for paste events to handle formatted text
$(editor).on('paste', function(e) {
e.preventDefault();

// Get pasted data via clipboard API
const clipboardData = e.originalEvent.clipboardData || window.clipboardData;
let pastedData = clipboardData.getData('text/html') || clipboardData.getData('text');

// Insert into editor
document.execCommand('insertHTML', false, pastedData);
saveCurrentState();
});

// Add keyboard shortcuts
$(document).on('keydown', function(e) {
if (e.ctrlKey || e.metaKey) {
switch (e.key.toLowerCase()) {
    case 'b':
        e.preventDefault();
        $('#boldBtn').click();
        break;
    case 'i':
        e.preventDefault();
        $('#italicBtn').click();
        break;
    case 'u':
        e.preventDefault();
        $('#underlineBtn').click();
        break;
    case 'z':
        e.preventDefault();
        if (e.shiftKey) {
            $('#redoBtn').click();
        } else {
            $('#undoBtn').click();
        }
        break;
    case 'y':
        e.preventDefault();
        $('#redoBtn').click();
        break;
}
}
});

// Create download function for saving content
function downloadContent(filename, content) {
const element = document.createElement('a');
element.setAttribute('href', 'data:text/html;charset=utf-8,' + encodeURIComponent(content));
element.setAttribute('download', filename);
element.style.display = 'none';
document.body.appendChild(element);
element.click();
document.body.removeChild(element);
}

// Add save button to toolbar
$('<div class="toolbar-group">' +
'<button id="saveBtn" title="Save Document"><i class="fas fa-save"></i></button>' +
'<button id="htmlBtn" title="View HTML"><i class="fas fa-code"></i></button>' +
'</div>').appendTo('.toolbar');

// Save button handler
$('#saveBtn').on('click', function() {
const content = `<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Rich Text Document</title>
<style>
body { font-family: Arial, sans-serif; line-height: 1.6; padding: 20px; }
table { border-collapse: collapse; width: 100%; }
th, td { border: 1px solid #ddd; padding: 8px; }
img { max-width: 100%; }
</style>
</head>
<body>
${editor.innerHTML}
</body>
</html>`;

downloadContent('document.html', content);
});

// HTML view toggle
let isHtmlView = false;
$('#htmlBtn').on('click', function() {
if (!isHtmlView) {
const htmlContent = editor.innerHTML;
editor.textContent = htmlContent;
$('#editorMode').text('HTML Mode');
isHtmlView = true;
} else {
const htmlContent = editor.textContent;
editor.innerHTML = htmlContent;
$('#editorMode').text('Design Mode');
isHtmlView = false;
MathJax.typesetPromise([editor]).catch(function(err) {
    console.log('MathJax error:', err);
});
}
});

// Initial call to update button states
updateButtonStates();



});

